//==============================================================================
// Project:		Wealth Transfers and their economic effects
// File name:	Figure 2.8
// Objective: 	Generate the modelling underpinning figure 2.8
//
//==============================================================================

clear all

************************
*** Data import ***
************************

cd "XXXX" //set content directory here

use combined_filtered_restricted.dta, replace

merge m:1 xwaveid wave using bequest_from_HH_death.dta //flag for inheritances due to HH deaths
drop if _merge == 2
drop _merge

xtset xwaveid wave

***********************************
*** Create equivalisation scale ***
***********************************

gen child = 1 if hgage < 15
gen adult = 1 if hgage >= 15

egen total_child = total(child), by(hhrhid wave)
egen total_adult = total(adult), by(hhrhid wave)

gen equiv_scale = total_child*0.3 + total_adult*0.5 + 0.5 //0.3 for children, 1 for first adult and 0.5 for all subsequent adults

drop if hgint == 0 //drop non-responders

***********************
*** Create variables **
***********************

replace oifinha = 0 if oifinha == -1 //inheritances
replace oifinha = . if oifinha < -1
gen bequest = oifinha
replace bequest = 0 if bequest > 0 & bequest != . & bequest_from_HH_death == 1 //filter out inheritances due to HH deaths

gen hgage_squ = hgage^2 //age squared

gen education = . //education
replace education = 1 if edhigh1 == 1 //postgrad
replace education = 2 if edhigh1 == 2 //grad dip or cert
replace education = 3 if edhigh1 == 3 //bachelor or honours
replace education = 4 if edhigh1 == 4 | edhigh1 == 5 //cert (3+) or dip
replace education = 5 if edhigh1 == 8 | edhigh1 == 9 //highschool

gen male = 1 if sex == 1 //gender
gen female = 1 if sex == 2
replace female = 0 if female == .
replace male = 0 if male == .

replace fmmhlq = 0 if fmmhlq == -1 //mother's level of education
replace fmmhlq = . if fmmhlq < -1
gen mother_education_5 = . //first asked in wave 5
replace mother_education_5 = fmmhlq if wave == 5
egen mother_education = min(mother_education_5), by(xwaveid)
replace mother_education = fmmhlq if wave > 5
replace mother_education = 2 if mother_education == 2 | mother_education == 3 | mother_education == 4  //group the VET options
replace mother_education = 3 if mother_education == 5
replace mother_education = 4 if mother_education == 8

replace fmfhlq = 0 if fmfhlq == -1 //father's level of education
replace fmfhlq = . if fmfhlq < -1
gen father_education_5 = . //first asked in wave 5
replace father_education_5 = fmfhlq if wave == 5
egen father_education = min(father_education_5), by(xwaveid)
replace father_education = fmfhlq if wave > 5
replace father_education = 2 if father_education == 2 | father_education == 3 | father_education == 4  //group the VET options
replace father_education = 3 if father_education == 5
replace father_education = 4 if father_education == 8

replace anbcob = . if anbcob < 0 //country of birth
gen cob = anbcob

gen living_with_parents_at_14 = . //living with parents at 14
replace living_with_parents_at_14 = 1 if fmlwop == 1 //both parents
replace living_with_parents_at_14 = 2 if fmlwop == 2 | fmlwop == 3 //one parent and one step parent 
replace living_with_parents_at_14 = 3 if fmlwop == 4 | fmlwop == 5 //single parent
replace living_with_parents_at_14 = 4 if fmlwop == 8 //other

replace tcr = . if tcr < 0 //number of resident children
gen resident_children = tcr

replace bsn = 0 if bsn == -1 //number of siblings
replace bsn = . if bsn < -1
egen siblings = max(bsn), by(xwaveid) //only asked in some waves - imputed for other waves

gen partner = 0 //partner
replace partner = 1 if hhpxid > 0 & hhpxid != .

gen net_worth = hwnwip - hwnwin //household net worth
gen delta_net_worth_4 = net_worth - L4.net_worth
gen delta_net_worth_8 = net_worth - L8.net_worth
gen net_worth_squ = net_worth^2

gen net_worth_e = net_worth / equiv_scale //equivalised net worth

gen HH_disposable_income = hifdip - hifdin //household disposable income

gen L6_4_bequest = L6.bequest + L5.bequest + L4.bequest //pooled inheritances
gen L2_0_bequest = L2.bequest + L.bequest + bequest


******************************
*** Create wealth variables **
******************************

gen bank_accounts = hwtbani
gen superannuation = hwsupei
gen other_non_fin_assets = hwnfii - hwtpvi

gen bonds = hwcaini
gen stocks = hweqini
gen trust_funds = hwtrusi
gen life_insurance = hwinsui
gen trust_life_insur = trust_funds + life_insurance

gen home = hwhmvai
gen investment_property = hwopvai

gen total_debts = hwdebti

local wealth_types bank_accounts superannuation bonds stocks trust_life_insur home investment_property other_non_fin_assets total_debts
foreach x of local wealth_types{
gen delta_`x'_4 = `x' - L4.`x'
gen delta_`x'_8 = `x' - L8.`x'
gen `x'_squ = `x'^2
}

***********************
*** Create weights ****
***********************

local weights_4 wlrb_g wlrf_k wlrj_o wlrn_s //balanced longitudinal weights from t-4 to t+1  

foreach x of local weights_4{
    egen total_`x' = total(`x'), by(wave)
	egen max_total_`x' = max(total_`x')
	gen `x'_normalised = 0 //need to normalise weights so that earlier and later years have equal weighting
	replace `x'_normalised = `x'/max_total_`x' if `x' > 0
}

gen weight_4 = 0 //all weights in single variable
replace weight_4 = wlrb_g_normalised if wave == 6
replace weight_4 = wlrf_k_normalised if wave == 10
replace weight_4 = wlrj_o_normalised if wave == 14
replace weight_4 = wlrn_s_normalised if wave == 18

local weights_8 wlrb_k wlrf_o wlrj_s //balanced longitudinal weights from t-8 to t+1 

foreach x of local weights_8{
    egen total_`x' = total(`x'), by(wave)
	egen max_total_`x' = max(total_`x')
	gen `x'_normalised = 0
	replace `x'_normalised = `x'/max_total_`x' if `x' > 0
}

gen weight_8 = 0 //all weights in single variable
replace weight_8 = wlrb_k_normalised if wave == 10
replace weight_8 = wlrf_o_normalised if wave == 14
replace weight_8 = wlrj_s_normalised if wave == 18


*****************
*** Models ******
*****************

local controls_parent_chars_4 i.wave hgage hgage_squ i.sex L4.resident_children L4.partner i.L4.education i.mother_education i.father_education i.cob i.living_with_parents_at_14 siblings L4.HH_disposable_income
local wealth_types bank_accounts superannuation bonds stocks trust_life_insur home investment_property other_non_fin_assets total_debts
foreach x of local wealth_types{
quietly reg delta_`x'_4 `controls_parent_chars_4' L2_0_bequest [pweight = weight_4] if weight_4 > 0 & ((L2_0_bequest > 0 & L2_0_bequest != .) | (F.bequest > 0 & F.bequest != .)) & wave != 18, vce(cluster xwaveid)
estimates store `x'
}

esttab bank_accounts superannuation bonds stocks trust_life_insur home investment_property other_non_fin_assets total_debts, ci(a3) keep(L2_0_bequest) scsv nostar noparentheses //figure 2.8, panel a

local controls_parent_chars_8 i.wave hgage hgage_squ i.sex L8.resident_children L8.partner i.L8.education i.mother_education i.father_education i.cob i.living_with_parents_at_14 siblings L8.HH_disposable_income
local wealth_types bank_accounts superannuation bonds stocks trust_life_insur home investment_property other_non_fin_assets total_debts
foreach x of local wealth_types{
reg delta_`x'_8 `controls_parent_chars_8' L6_4_bequest [pweight = weight_8] if weight_8 > 0 & ((L6_4_bequest > 0 & L6_4_bequest != .) | (F.bequest > 0 & F.bequest != .)), vce(cluster xwaveid)
estimates store `x'
}

esttab bank_accounts superannuation bonds stocks trust_life_insur home investment_property other_non_fin_assets total_debts, ci(a3) keep(L6_4_bequest) scsv nostar noparentheses //figure 2.8, panel b

